<?php
declare(strict_types=1);

/*
|--------------------------------------------------------------------------
| ZENTRALE KONFIG
|--------------------------------------------------------------------------
*/
function app_config(): array {
    $cfgFile = __DIR__ . '/../config.php';
    if (!file_exists($cfgFile)) {
        return [
            'installed' => false,
            'app_name'  => 'Bibliothek',
            'db_path'   => __DIR__ . '/../data/library.db',
        ];
    }
    return require $cfgFile;
}

/*
|--------------------------------------------------------------------------
| SESSION – EDGE-SICHER & ROBUST
|--------------------------------------------------------------------------
*/
function ensure_session(): void
{
    if (session_status() === PHP_SESSION_ACTIVE) {
        return;
    }

    // HTTPS zuverlässig erkennen
    $isHttps = (
        (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')
        || (!empty($_SERVER['SERVER_PORT']) && (int)$_SERVER['SERVER_PORT'] === 443)
    );

    session_name('LIBSESSID'); // eigener Name → keine Kollisionen

    session_set_cookie_params([
        'lifetime' => 0,
        'path'     => '/',
        'domain'   => '',        // leer lassen → Edge-freundlich
        'secure'   => $isHttps,  // NUR bei HTTPS true
        'httponly' => true,
        'samesite' => 'Lax',     // korrekt für Mobirise + include
    ]);

    session_start();
}

/*
|--------------------------------------------------------------------------
| DATENBANK (SQLite)
|--------------------------------------------------------------------------
*/
function db(): PDO {
    static $pdo = null;
    if ($pdo) return $pdo;

    $cfg  = app_config();
    $path = $cfg['db_path'] ?? (__DIR__ . '/../data/library.db');

    if (!is_file($path)) {
        throw new RuntimeException('Datenbank nicht gefunden. Bitte Installer ausführen.');
    }

    $pdo = new PDO('sqlite:' . $path, null, null, [
        PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);

    $pdo->exec("PRAGMA foreign_keys = ON;");
    return $pdo;
}

/*
|--------------------------------------------------------------------------
| JSON HELFER
|--------------------------------------------------------------------------
*/
function json_response($data, int $code = 200): void {
    http_response_code($code);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

function read_json_body(): array {
    $raw = file_get_contents('php://input');
    if ($raw === false || trim($raw) === '') return [];
    $data = json_decode($raw, true);
    return is_array($data) ? $data : [];
}

/*
|--------------------------------------------------------------------------
| INSTALLATIONS-CHECK
|--------------------------------------------------------------------------
*/
function require_installed(): void {
    $cfg = app_config();
    if (empty($cfg['installed'])) {
        json_response([
            'ok'    => false,
            'error' => 'Nicht installiert. Bitte /install/ ausführen.'
        ], 400);
    }
}

/*
|--------------------------------------------------------------------------
| AUTH / USER
|--------------------------------------------------------------------------
*/
function current_user(): ?array {
    ensure_session();
    if (empty($_SESSION['user_id'])) return null;

    return [
        'id'   => (int)$_SESSION['user_id'],
        'name' => (string)($_SESSION['user_name'] ?? ''),
        'role' => (string)($_SESSION['role'] ?? 'member'),
    ];
}

function require_login(): array {
    $u = current_user();
    if (!$u) {
        json_response(['ok' => false, 'error' => 'Bitte zuerst einloggen.'], 401);
    }
    return $u;
}

function require_admin(): array {
    $u = require_login();
    if ($u['role'] !== 'admin') {
        json_response(['ok' => false, 'error' => 'Admin-Rechte erforderlich.'], 403);
    }
    return $u;
}

/*
|--------------------------------------------------------------------------
| BASE URL (für Bilder etc.)
|--------------------------------------------------------------------------
*/
function base_url(): string {
    // ermittelt z.B. /buch oder /buch1
    $basePath = dirname(__DIR__);
    $docRoot  = rtrim($_SERVER['DOCUMENT_ROOT'], '/');
    $url      = str_replace($docRoot, '', $basePath);
    return $url !== '' ? $url : '/';
}
