<?php
declare(strict_types=1);
require __DIR__ . '/../api/db.php';
require_installed();
ensure_session();
$cfg = app_config();
$appName = htmlspecialchars((string)($cfg['app_name'] ?? 'Bibliothek'), ENT_QUOTES, 'UTF-8');
?>
<style>
  #libapp * { box-sizing: border-box; }
  #libapp {
    max-width: 980px;
    margin: 0 auto;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
    color: #1a1a1a;
  }
  #libapp h2 {
    margin: 0;
    font-size: 28px;
    font-weight: 700;
    color: #0f172a;
  }
  #libapp h3 {
    margin: 0;
    font-size: 20px;
    font-weight: 600;
    color: #334155;
  }
  .header-box {
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 12px;
    flex-wrap: wrap;
    margin-bottom: 20px;
  }
  #meBox {
    font-size: 14px;
    color: #475569;
    padding: 8px 16px;
    background: #f1f5f9;
    border-radius: 8px;
  }
  #meBox strong {
    color: #0f172a;
    font-weight: 600;
  }
  #loginBox {
    margin-top: 14px;
    padding: 20px;
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    border: 1px solid #e2e8f0;
    border-radius: 16px;
    box-shadow: 0 1px 3px rgba(0,0,0,0.05);
  }
  .login-form {
    display: flex;
    gap: 12px;
    flex-wrap: wrap;
    align-items: end;
  }
  .input-group {
    flex: 1;
    min-width: 220px;
  }
  .input-group label {
    display: block;
    font-size: 13px;
    font-weight: 500;
    color: #64748b;
    margin-bottom: 6px;
  }
  #nameInput, #searchInput {
    width: 100%;
    padding: 12px 14px;
    border-radius: 10px;
    border: 1px solid #cbd5e1;
    font-size: 15px;
    transition: all 0.2s;
    background: white;
  }
  #nameInput:focus, #searchInput:focus {
    outline: none;
    border-color: #3b82f6;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
  }
  .btn {
    padding: 12px 20px;
    border-radius: 10px;
    border: 0;
    cursor: pointer;
    font-weight: 600;
    font-size: 14px;
    transition: all 0.2s;
    white-space: nowrap;
  }
  #loginBtn {
    background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%);
    color: white;
    box-shadow: 0 2px 8px rgba(59, 130, 246, 0.3);
  }
  #loginBtn:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(59, 130, 246, 0.4);
  }
  #logoutBtn {
    background: #ef4444;
    color: white;
    box-shadow: 0 2px 8px rgba(239, 68, 68, 0.3);
  }
  #logoutBtn:hover {
    background: #dc2626;
    transform: translateY(-1px);
  }
  #reloadBtn {
    background: white;
    border: 1px solid #e2e8f0;
    color: #475569;
    box-shadow: 0 1px 3px rgba(0,0,0,0.05);
  }
  #reloadBtn:hover {
    background: #f8fafc;
    border-color: #cbd5e1;
  }
  #loginMsg {
    margin-top: 12px;
    font-size: 14px;
    padding: 10px 14px;
    border-radius: 8px;
    font-weight: 500;
  }
  #loginMsg:not(:empty) {
    background: #dbeafe;
    color: #1e40af;
  }
  .search-box {
    margin: 24px 0 16px 0;
    padding: 16px;
    background: white;
    border: 1px solid #e2e8f0;
    border-radius: 16px;
    box-shadow: 0 1px 3px rgba(0,0,0,0.05);
  }
  .search-controls {
    display: flex;
    gap: 12px;
    align-items: center;
    flex-wrap: wrap;
  }
  .search-controls .input-group {
    flex: 1;
    min-width: 300px;
  }
  .books-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 12px;
    flex-wrap: wrap;
    margin: 24px 0 16px 0;
  }
  .book-card {
    display: flex;
    gap: 16px;
    justify-content: space-between;
    align-items: center;
    padding: 16px;
    background: white;
    border: 1px solid #e2e8f0;
    border-radius: 16px;
    margin-bottom: 12px;
    flex-wrap: wrap;
    transition: all 0.2s;
    box-shadow: 0 1px 3px rgba(0,0,0,0.05);
  }
  .book-card:hover {
    box-shadow: 0 4px 12px rgba(0,0,0,0.08);
    transform: translateY(-2px);
  }
  .book-info {
    display: flex;
    gap: 16px;
    align-items: flex-start;
    min-width: 260px;
    flex: 1;
  }
  .book-cover {
    width: 80px;
    height: 110px;
    object-fit: cover;
    border-radius: 12px;
    border: 1px solid #e2e8f0;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
  }
  .book-details {
    flex: 1;
  }
  .book-title {
    font-weight: 600;
    font-size: 16px;
    color: #0f172a;
    margin-bottom: 4px;
  }
  .book-meta {
    font-size: 13px;
    color: #64748b;
    margin-bottom: 2px;
  }
  .book-note {
    font-size: 13px;
    color: #64748b;
    margin-top: 6px;
    font-style: italic;
  }
  .book-actions {
    display: flex;
    gap: 12px;
    align-items: center;
    flex-wrap: wrap;
  }
  .status-badge {
    display: inline-block;
    padding: 6px 14px;
    border-radius: 999px;
    font-size: 12px;
    font-weight: 600;
    white-space: nowrap;
  }
  .status-free {
    background: #d1fae5;
    color: #065f46;
  }
  .status-loaned {
    background: #fef3c7;
    color: #92400e;
  }
  .loan-info {
    font-size: 12px;
    color: #64748b;
  }
  .loan-btn {
    padding: 10px 16px;
    border-radius: 10px;
    border: 0;
    cursor: pointer;
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
    font-weight: 600;
    font-size: 13px;
    transition: all 0.2s;
    box-shadow: 0 2px 8px rgba(16, 185, 129, 0.3);
  }
  .loan-btn:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(16, 185, 129, 0.4);
  }
  .pagination {
    display: flex;
    gap: 8px;
    align-items: center;
    justify-content: center;
    margin: 24px 0;
    flex-wrap: wrap;
  }
  .pagination button {
    padding: 8px 14px;
    border-radius: 8px;
    border: 1px solid #e2e8f0;
    background: white;
    color: #475569;
    cursor: pointer;
    font-size: 14px;
    font-weight: 500;
    transition: all 0.2s;
  }
  .pagination button:hover:not(:disabled) {
    background: #f8fafc;
    border-color: #cbd5e1;
  }
  .pagination button:disabled {
    opacity: 0.5;
    cursor: not-allowed;
  }
  .pagination button.active {
    background: #3b82f6;
    color: white;
    border-color: #3b82f6;
  }
  .pagination-info {
    font-size: 14px;
    color: #64748b;
    padding: 0 12px;
  }
  .empty-state {
    padding: 40px 20px;
    text-align: center;
    color: #94a3b8;
    background: #f8fafc;
    border-radius: 16px;
    border: 1px dashed #cbd5e1;
  }
  .loading {
    padding: 20px;
    text-align: center;
    color: #64748b;
  }
</style>

<div id="libapp">
  <div class="header-box">
    <h2><?= $appName ?> – Ausleihe</h2>
    <div id="meBox"></div>
  </div>

  <div id="loginBox">
    <div class="login-form">
      <div class="input-group">
        <label>Dein Name</label>
        <input id="nameInput" placeholder="z.B. Max Mustermann">
      </div>
      <button id="loginBtn" class="btn">Einloggen</button>
      <button id="logoutBtn" class="btn" style="display:none;">Logout</button>
    </div>
    <div id="loginMsg"></div>
  </div>

  <div class="search-box">
    <div class="search-controls">
      <div class="input-group">
        <label>🔍 Bücher durchsuchen</label>
        <input id="searchInput" placeholder="Titel, Autor oder ISBN suchen...">
      </div>
      <button id="reloadBtn" class="btn">🔄 Aktualisieren</button>
    </div>
  </div>

  <div class="books-header">
    <h3>Bücher</h3>
    <div id="resultsInfo" style="font-size: 14px; color: #64748b;"></div>
  </div>
  <div id="booksBox"></div>
  <div id="paginationBox" class="pagination"></div>
</div>

<script>
(function(){
  const apiBase = "/test123/api"; // Pfad hier anpassen !!!
  const ITEMS_PER_PAGE = 4; // Anzahl hier einstellen

  let allBooks = [];
  let filteredBooks = [];
  let currentPage = 1;

  const el = (id)=>document.getElementById(id);
  const meBox = el('meBox');
  const loginMsg = el('loginMsg');
  const loginBtn = el('loginBtn');
  const logoutBtn = el('logoutBtn');
  const reloadBtn = el('reloadBtn');
  const booksBox = el('booksBox');
  const paginationBox = el('paginationBox');
  const searchInput = el('searchInput');
  const resultsInfo = el('resultsInfo');

  function esc(s){ return String(s ?? '').replace(/[&<>"']/g, m=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;' }[m])); }

  async function api(path, body){
    const opt = body ? {method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify(body)} : {};
    const res = await fetch(apiBase + '/' + path, opt);
    return await res.json();
  }

  async function refreshMe(){
    const r = await api('me.php');
    const u = r.user;
    if (u && u.name) {
      meBox.innerHTML = 'Eingeloggt als <strong>'+esc(u.name)+'</strong>';
      logoutBtn.style.display = '';
      loginBtn.style.display = 'none';
      el('nameInput').style.display = 'none';
      el('nameInput').parentElement.style.display = 'none';
    } else {
      meBox.innerHTML = '';
      logoutBtn.style.display = 'none';
      loginBtn.style.display = '';
      el('nameInput').style.display = '';
      el('nameInput').parentElement.style.display = '';
    }
  }

  function bookRow(b){
    const free = Number(b.available) === 1;
    const statusHtml = free
      ? '<span class="status-badge status-free">✓ Verfügbar</span>'
      : '<span class="status-badge status-loaned">● Ausgeliehen</span>';

    const actionHtml = free
      ? '<button data-loan="'+esc(b.id)+'" class="loan-btn">📚 Ausleihen</button>'
      : '<span class="loan-info">von '+esc(b.loaned_to || '—')+'</span>';

    const img = esc(b.image_url || '');
    return `
      <div class="book-card">
        <div class="book-info">
          <img src="${img}" alt="" class="book-cover">
          <div class="book-details">
            <div class="book-title">${esc(b.title)}</div>
            <div class="book-meta">${esc(b.author || '')}${b.isbn ? ' • ISBN '+esc(b.isbn) : ''}</div>
            ${b.note ? '<div class="book-note">'+esc(b.note)+'</div>' : ''}
          </div>
        </div>
        <div class="book-actions">
          ${statusHtml}
          ${actionHtml}
        </div>
      </div>
    `;
  }

  function filterBooks(searchTerm) {
    if (!searchTerm) {
      filteredBooks = [...allBooks];
    } else {
      const term = searchTerm.toLowerCase();
      filteredBooks = allBooks.filter(b => 
        (b.title || '').toLowerCase().includes(term) ||
        (b.author || '').toLowerCase().includes(term) ||
        (b.isbn || '').toLowerCase().includes(term)
      );
    }
    currentPage = 1;
    renderBooks();
  }

  function renderBooks() {
    const totalPages = Math.ceil(filteredBooks.length / ITEMS_PER_PAGE);
    const startIdx = (currentPage - 1) * ITEMS_PER_PAGE;
    const endIdx = startIdx + ITEMS_PER_PAGE;
    const pageBooks = filteredBooks.slice(startIdx, endIdx);

    if (filteredBooks.length === 0) {
      booksBox.innerHTML = '<div class="empty-state">📚 Keine Bücher gefunden.</div>';
      paginationBox.innerHTML = '';
      resultsInfo.textContent = '';
      return;
    }

    booksBox.innerHTML = pageBooks.map(bookRow).join('');
    
    resultsInfo.textContent = `${filteredBooks.length} Buch${filteredBooks.length !== 1 ? 'ü' : ''}cher${searchInput.value ? ' gefunden' : ''}`;

    booksBox.querySelectorAll('button[data-loan]').forEach(btn=>{
      btn.addEventListener('click', async ()=>{
        const bookId = Number(btn.getAttribute('data-loan'));
        loginMsg.textContent = '';
        const res = await api('loan.php', {book_id: bookId});
        loginMsg.textContent = res.ok ? '✓ Ausleihe gespeichert.' : (res.error || 'Fehler');
        await loadBooks();
        await refreshMe();
      });
    });

    renderPagination(totalPages);
  }

  function renderPagination(totalPages) {
    if (totalPages <= 1) {
      paginationBox.innerHTML = '';
      return;
    }

    let html = '';
    
    html += `<button ${currentPage === 1 ? 'disabled' : ''} data-page="${currentPage - 1}">← Zurück</button>`;
    
    const maxButtons = 7;
    let startPage = Math.max(1, currentPage - Math.floor(maxButtons / 2));
    let endPage = Math.min(totalPages, startPage + maxButtons - 1);
    
    if (endPage - startPage < maxButtons - 1) {
      startPage = Math.max(1, endPage - maxButtons + 1);
    }

    if (startPage > 1) {
      html += `<button data-page="1">1</button>`;
      if (startPage > 2) html += `<span class="pagination-info">...</span>`;
    }

    for (let i = startPage; i <= endPage; i++) {
      html += `<button class="${i === currentPage ? 'active' : ''}" data-page="${i}">${i}</button>`;
    }

    if (endPage < totalPages) {
      if (endPage < totalPages - 1) html += `<span class="pagination-info">...</span>`;
      html += `<button data-page="${totalPages}">${totalPages}</button>`;
    }

    html += `<button ${currentPage === totalPages ? 'disabled' : ''} data-page="${currentPage + 1}">Weiter →</button>`;

    paginationBox.innerHTML = html;

    paginationBox.querySelectorAll('button[data-page]').forEach(btn => {
      btn.addEventListener('click', () => {
        currentPage = Number(btn.getAttribute('data-page'));
        renderBooks();
        window.scrollTo({ top: 0, behavior: 'smooth' });
      });
    });
  }

  async function loadBooks(){
    booksBox.innerHTML = '<div class="loading">Laden…</div>';
    paginationBox.innerHTML = '';
    resultsInfo.textContent = '';
    
    const r = await api('books.php');
    if (!r.ok) {
      booksBox.innerHTML = '<div class="book-card" style="background:#fee;border-color:#fcc;">Fehler: '+esc(r.error)+'</div>';
      return;
    }
    
    allBooks = r.books || [];
    filterBooks(searchInput.value);
  }

  searchInput.addEventListener('input', (e) => {
    filterBooks(e.target.value);
  });

  loginBtn.addEventListener('click', async ()=>{
    loginMsg.textContent = '';
    const name = (el('nameInput').value || '').trim();
    if (!name) { loginMsg.textContent = 'Bitte Name eingeben.'; return; }
    const r = await api('login.php', {name});
    loginMsg.textContent = r.ok ? '✓ Eingeloggt.' : (r.error || 'Fehler');
    await refreshMe();
  });

  logoutBtn.addEventListener('click', async ()=>{
    await api('logout.php');
    await refreshMe();
    loginMsg.textContent = 'Ausgeloggt.';
  });

  reloadBtn.addEventListener('click', loadBooks);

  refreshMe().then(loadBooks);
})();
</script>