<?php
// backup.php - Einfaches Backup-System für das Gästebuch
// Nur für Admins zugänglich!

session_start();

// Sicherheitscheck
if (!isset($_SESSION['admin'])) {
    die('Zugriff verweigert');
}

$backup_dir = 'backups/';
if (!is_dir($backup_dir)) {
    mkdir($backup_dir, 0755, true);
}

function createBackup() {
    global $backup_dir;
    
    $timestamp = date('Y-m-d_H-i-s');
    $backup_file = $backup_dir . "guestbook_backup_$timestamp.zip";
    
    $zip = new ZipArchive();
    
    if ($zip->open($backup_file, ZipArchive::CREATE) === TRUE) {
        // JSON-Dateien hinzufügen
        if (file_exists('entries.json')) {
            $zip->addFile('entries.json');
        }
        if (file_exists('pending.json')) {
            $zip->addFile('pending.json');
        }
        
        // Konfiguration hinzufügen (falls vorhanden)
        if (file_exists('config.php')) {
            $zip->addFile('config.php');
        }
        
        $zip->close();
        return $backup_file;
    }
    
    return false;
}

function listBackups() {
    global $backup_dir;
    
    $backups = [];
    if (is_dir($backup_dir)) {
        $files = scandir($backup_dir);
        foreach ($files as $file) {
            if (pathinfo($file, PATHINFO_EXTENSION) === 'zip') {
                $backups[] = [
                    'file' => $file,
                    'path' => $backup_dir . $file,
                    'size' => filesize($backup_dir . $file),
                    'date' => filemtime($backup_dir . $file)
                ];
            }
        }
        // Nach Datum sortieren (neueste zuerst)
        usort($backups, function($a, $b) {
            return $b['date'] - $a['date'];
        });
    }
    
    return $backups;
}

// Backup erstellen
if (isset($_POST['create_backup'])) {
    $backup_file = createBackup();
    if ($backup_file) {
        $message = "Backup erfolgreich erstellt: " . basename($backup_file);
    } else {
        $error = "Fehler beim Erstellen des Backups";
    }
}

// Backup herunterladen
if (isset($_GET['download']) && file_exists($_GET['download'])) {
    $file = $_GET['download'];
    header('Content-Type: application/zip');
    header('Content-Disposition: attachment; filename="' . basename($file) . '"');
    header('Content-Length: ' . filesize($file));
    readfile($file);
    exit;
}

// Backup löschen
if (isset($_POST['delete_backup']) && file_exists($_POST['backup_file'])) {
    unlink($_POST['backup_file']);
    $message = "Backup wurde gelöscht";
}

$backups = listBackups();
?>

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Backup-Verwaltung</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap-icons/1.10.0/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        body {
            margin-bottom: 50px;
        }
    </style>    
    <!-- Resize-Skript von Werner Zenk  -->
    <script>
        window.addEventListener("load", fenster);
        window.addEventListener("resize", fenster);
        function fenster() {
          if (parent.document.getElementsByTagName("iframe")[0]) {
            parent.document.getElementsByTagName("iframe")[0].style.height =
              document.getElementsByTagName("html")[0].offsetHeight +'px';
          }
        }
    </script>    
</head>
<body class="bg-light">
    <div class="container mt-5">
        <div class="row justify-content-center">
            <div class="col-md-12">
                <div class="card">
                    <div class="card-header bg-info text-white">
                        <h4><i class="bi bi-archive"></i> Backup-Verwaltung</h4>
                    </div>
                    <div class="card-body">
                        <!-- Nachrichten -->
                        <?php if (isset($message)): ?>
                            <div class="alert alert-success"><?= $message ?></div>
                        <?php endif; ?>
                        
                        <?php if (isset($error)): ?>
                            <div class="alert alert-danger"><?= $error ?></div>
                        <?php endif; ?>

                        <!-- Neues Backup erstellen -->
                        <form method="POST" class="mb-4">
                            <button type="submit" name="create_backup" class="btn btn-primary">
                                <i class="bi bi-plus-circle"></i> Neues Backup erstellen
                            </button>
                        </form>

                        <!-- Backup-Liste -->
                        <h5>Vorhandene Backups</h5>
                        <?php if (empty($backups)): ?>
                            <p class="text-muted">Keine Backups vorhanden</p>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-striped">
                                    <thead>
                                        <tr>
                                            <th>Datei</th>
                                            <th>Datum</th>
                                            <th>Größe</th>
                                            <th>Aktionen</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($backups as $backup): ?>
                                            <tr>
                                                <td><?= htmlspecialchars($backup['file']) ?></td>
                                                <td><?= date('d.m.Y H:i', $backup['date']) ?></td>
                                                <td><?= number_format($backup['size'] / 1024, 2) ?> KB</td>
                                                <td>
                                                    <a href="?download=<?= urlencode($backup['path']) ?>" 
                                                       class="btn btn-sm btn-success" style="width: 100px;">
                                                        <i class="bi bi-download"></i> Download
                                                    </a>
                                                    <form method="POST" class="d-inline">
                                                        <input type="hidden" name="backup_file" value="<?= $backup['path'] ?>">
                                                        <button type="submit" name="delete_backup" 
                                                                style="width: 100px; margin-top: 5px;"
                                                                class="btn btn-sm btn-danger"
                                                                onclick="return confirm('Backup wirklich löschen?')">
                                                            <i class="bi bi-trash"></i> Löschen
                                                        </button>
                                                    </form>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>

                        <div class="mt-4">
                            <a href="admin.php" class="btn btn-secondary">
                                <i class="bi bi-arrow-left"></i> Zurück zur Administration
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/js/bootstrap.bundle.min.js"></script>
</body>
</html>