<?php
session_start();

// Konfiguration für Paginierung
$ENTRIES_PER_PAGE = 5;

// Konfiguration
$entries_file = 'entries.json';

// SICHERHEITS-OPTIONEN für E-Mail-Anzeige:
// 1 = E-Mails komplett verstecken
// 2 = E-Mails verschleiern (m***@example.com)
// 3 = E-Mails nur für Admin sichtbar
// 4 = Gravatar-Avatar anzeigen statt E-Mail
$EMAIL_DISPLAY_MODE = 3;

// Hilfsfunktionen
function loadEntries($file) {
    if (!file_exists($file)) {
        return [];
    }
    $content = file_get_contents($file);
    return json_decode($content, true) ?: [];
}

// E-Mail verschleiern
function obfuscateEmail($email) {
    if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        return '';
    }
    
    $parts = explode('@', $email);
    $name = $parts[0];
    $domain = $parts[1];
    
    // Ersten und letzten Buchstaben zeigen, Rest mit Sternen
    if (strlen($name) <= 2) {
        $obfuscated_name = str_repeat('*', strlen($name));
    } else {
        $obfuscated_name = substr($name, 0, 1) . str_repeat('*', strlen($name) - 2) . substr($name, -1);
    }
    
    return $obfuscated_name . '@' . $domain;
}

// Gravatar URL generieren (falls gewünscht)
function getGravatar($email, $size = 40) {
    if (empty($email)) return '';
    $hash = md5(strtolower(trim($email)));
    return "https://www.gravatar.com/avatar/$hash?s=$size&d=identicon&r=g";
}

// Daten laden
$entries = loadEntries($entries_file);

// Einträge nach Datum sortieren (neueste zuerst)
usort($entries, function($a, $b) {
    return strtotime($b['date']) - strtotime($a['date']);
});

// Paginierungs-Variablen
$total_entries = count($entries);
$total_pages = ceil($total_entries / $ENTRIES_PER_PAGE);

// Aktuelle Seite festlegen (standardmäßig 1)
$current_page = isset($_GET['page']) && is_numeric($_GET['page']) && $_GET['page'] > 0 ? (int)$_GET['page'] : 1;
if ($current_page > $total_pages) {
    $current_page = $total_pages > 0 ? $total_pages : 1;
}

// Offset für array_slice berechnen
$offset = ($current_page - 1) * $ENTRIES_PER_PAGE;

// Prüfen ob Admin eingeloggt ist (vereinfacht)
$is_admin = isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true;
?>

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gästebuch</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap-icons/1.10.0/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        .entry-card {
            border-left: 4px solid #007bff;
            transition: transform 0.2s ease-in-out;
        }
        .entry-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
        }
        .gradient-bg {
            background: linear-gradient(135deg, #506AEB 0%, #152EB0 100%);
        }
        .form-card {
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            border: none;
        }
        .gravatar {
            border-radius: 50%;
            margin-right: 8px;
        }
        .email-hidden {
            color: #6c757d;
            font-style: italic;
        }
    </style>
    <!-- Resize-Skript  -->
    <script>
        window.addEventListener("load", fenster);
        window.addEventListener("resize", fenster);
        function fenster() {
          if (parent.document.getElementsByTagName("iframe")[0]) {
            parent.document.getElementsByTagName("iframe")[0].style.height =
              document.getElementsByTagName("html")[0].offsetHeight +'px';
          }
        }
    </script>
</head>
<body class="bg-light">
    <div class="gradient-bg text-white py-5 mb-5">
        <div class="container">
            <div class="row justify-content-center text-center">
                <div class="col-lg-8">
                    <h1 class="display-4 fw-bold mb-3">
                        <i class="bi bi-journal-text"></i> Gästebuch
                    </h1>
                    <p class="lead">Teilen Sie Ihre Gedanken und Erfahrungen mit uns!</p>
                </div>
            </div>
        </div>
    </div>

    <div class="container my-5">
        <div class="row justify-content-center">
            <div class="col-lg-8">
                <div class="d-flex justify-content-between mb-4">
                    <a href="eintragsformular.php" class="btn btn-primary">
                        <i class="bi bi-pencil-square"></i> Eintrag verfassen
                    </a>
                    <a href="admin.php" class="btn btn-outline-secondary">
                        <i class="bi bi-shield-lock"></i> Admin-Bereich
                    </a>
                </div>

                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h3 class="text-dark"><i class="bi bi-chat-dots"></i> Gästebucheinträge</h3>
                    <span class="badge bg-primary fs-6"><?= $total_entries ?> Einträge</span>
                </div>

                <?php if ($total_entries === 0): ?>
                    <div class="text-center py-5">
                        <div class="mb-4">
                            <i class="bi bi-journal display-1 text-muted"></i>
                        </div>
                        <h4 class="text-muted mt-3">Noch keine Einträge vorhanden</h4>
                        <p class="text-muted">Seien Sie der Erste und <a href="eintragsformular.php">hinterlassen Sie einen Eintrag!</a></p>
                    </div>
                <?php else: ?>
                    <div class="row">
                        <?php 
                        // Einträge für die aktuelle Seite abrufen
                        $display_entries = array_slice($entries, $offset, $ENTRIES_PER_PAGE);
                        foreach ($display_entries as $entry): 
                        ?>
                            <div class="col-12 mb-4">
                                <div class="card entry-card h-100">
                                    <div class="card-body">
                                        <div class="d-flex justify-content-between align-items-start mb-3">
                                            <div class="d-flex align-items-center">
                                                <?php if ($EMAIL_DISPLAY_MODE == 4 && !empty($entry['email'])): ?>
                                                    <!-- Gravatar-Avatar anzeigen -->
                                                    <img src="<?= getGravatar($entry['email']) ?>" 
                                                         alt="Avatar" class="gravatar" width="40" height="40">
                                                <?php endif; ?>
                                                
                                                <div>
                                                    <h5 class="card-title text-primary mb-1">
                                                        <i class="bi bi-person-circle"></i> <?= htmlspecialchars($entry['name']) ?>
                                                    </h5>
                                                    
                                                    <?php if (!empty($entry['email'])): ?>
                                                        <?php if ($EMAIL_DISPLAY_MODE == 1): ?>
                                                            <!-- E-Mail komplett verstecken -->
                                                            
                                                        <?php elseif ($EMAIL_DISPLAY_MODE == 2): ?>
                                                            <!-- E-Mail verschleiert anzeigen -->
                                                            <small class="text-muted d-block">
                                                                <i class="bi bi-envelope"></i> <?= htmlspecialchars(obfuscateEmail($entry['email'])) ?>
                                                            </small>
                                                            
                                                        <?php elseif ($EMAIL_DISPLAY_MODE == 3): ?>
                                                            <!-- E-Mail nur für Admin sichtbar -->
                                                            <?php if ($is_admin): ?>
                                                                <small class="text-muted d-block">
                                                                    <i class="bi bi-envelope"></i> <?= htmlspecialchars($entry['email']) ?>
                                                                    <span class="badge bg-warning ms-1">Admin</span>
                                                                </small>
                                                            <?php else: ?>
                                                                <small class="email-hidden d-block">
                                                                    <i class="bi bi-envelope"></i> E-Mail privat
                                                                </small>
                                                            <?php endif; ?>
                                                            
                                                        <?php elseif ($EMAIL_DISPLAY_MODE == 4): ?>
                                                            <!-- Nur Avatar, keine E-Mail-Adresse -->
                                                            
                                                        <?php else: ?>
                                                            <!-- Standard: E-Mail vollständig anzeigen (NICHT empfohlen!) -->
                                                            <small class="text-muted d-block">
                                                                <i class="bi bi-envelope"></i> <?= htmlspecialchars($entry['email']) ?>
                                                            </small>
                                                        <?php endif; ?>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                            <div class="text-end">
                                                <small class="text-muted">
                                                    <i class="bi bi-calendar3"></i> <?= htmlspecialchars($entry['date']) ?>
                                                </small>
                                            </div>
                                        </div>
                                        <hr class="my-3">
                                        <p class="card-text"><?= nl2br(htmlspecialchars($entry['comment'])) ?></p>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>

                    <nav aria-label="Gästebuch Paginierung" class="mt-4">
                        <ul class="pagination justify-content-center">
                            <li class="page-item <?= $current_page <= 1 ? 'disabled' : '' ?>">
                                <a class="page-link" href="?page=<?= max(1, $current_page - 1) ?>">Vorherige</a>
                            </li>

                            <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                                <li class="page-item <?= $current_page === $i ? 'active' : '' ?>">
                                    <a class="page-link" href="?page=<?= $i ?>"><?= $i ?></a>
                                </li>
                            <?php endfor; ?>

                            <li class="page-item <?= $current_page >= $total_pages ? 'disabled' : '' ?>">
                                <a class="page-link" href="?page=<?= min($total_pages, $current_page + 1) ?>">Nächste</a>
                            </li>
                        </ul>
                    </nav>
                <?php endif; ?>

                <div class="text-center mt-5 pt-4 border-top">
                    <p class="text-muted">
                        <i class="bi bi-heart-fill text-danger"></i> 
                        Vielen Dank für Ihren Besuch!
                        <br>
                        <small>E-Mail-Schutz: Modus <?= $EMAIL_DISPLAY_MODE ?> aktiv</small>
                    </p>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/js/bootstrap.bundle.min.js"></script>
    <!-- Sprunganker: im Parent zum #Gastbuch scrollen (für iFrame mit id="myFrame") -->
    <script>
    (function () {
      function scrollParentToGastbuch() {
        try {
          if (window.parent && window.parent !== window) {
            var parentDoc = window.parent.document;
            setTimeout(function () {
              var anchor = parentDoc.getElementById('Gastbuch');
              if (anchor) {
                anchor.scrollIntoView({ behavior: 'smooth', block: 'start' });
              }
            }, 50);
          }
        } catch (e) {
          console.warn('Scroll zu #Gastbuch nicht möglich:', e);
        }
      }

      window.addEventListener('load', scrollParentToGastbuch);

      document.addEventListener('click', function (e) {
        var link = e.target.closest('a.page-link');
        if (!link) return;

        var href = link.getAttribute('href') || '';
        if (href.indexOf('?page=') !== -1) {
          scrollParentToGastbuch();
        }
      });
    })();
    </script>
</body>
</html>