<?php
require_once __DIR__ . '/FileClientInterface.php';

class FtpClient implements FileClientInterface
{
    private $host;
    private $user;
    private $pass;
    private $port;
    private $conn;

    public function __construct(string $host, string $user, string $pass, int $port = 21)
    {
        $this->host = $host;
        $this->user = $user;
        $this->pass = $pass;
        $this->port = $port;
    }

    public function connect()
    {
        if ($this->conn) return;

        $this->conn = ftp_connect($this->host, $this->port, 10);
        if (!$this->conn) {
            throw new Exception("FTP-Verbindung fehlgeschlagen");
        }

        if (!ftp_login($this->conn, $this->user, $this->pass)) {
            throw new Exception("FTP-Login fehlgeschlagen");
        }

        ftp_pasv($this->conn, true);
    }

    /** ============================
     *  LIST DIRECTORY (mit Datum+Größe)
     *  ============================ */
    public function listDirectory(string $path): array
    {
        $this->connect();
        if ($path === '') $path = '.';

        $raw = ftp_rawlist($this->conn, $path);
        if (!is_array($raw)) return [];

        $result = [];

        foreach ($raw as $line) {

            // Format:
            // drwxr-xr-x 2 user group 4096 Jan 31 12:55 folder
            $parts = preg_split('/\s+/', $line, 9);
            if (count($parts) < 9) continue;

            $permissions = $parts[0];
            $size        = (int)$parts[4];
            $month       = $parts[5];
            $day         = $parts[6];
            $timeOrYear  = $parts[7];
            $name        = $parts[8];

            if ($name === '.' || $name === '..') continue;

            $isDir = $permissions[0] === 'd';
            $mtime = $this->parseDate($month, $day, $timeOrYear);

            $result[] = [
                'name'   => $name,
                'is_dir' => $isDir,
                'size'   => $isDir ? 0 : $size,
                'mtime'  => $mtime
            ];
        }

        return $result;
    }

    private function parseDate($month, $day, $timeOrYear): int
    {
        $monthNum = date('n', strtotime($month));

        if (str_contains($timeOrYear, ':')) {
            $year = date('Y'); // gleiche Jahr
            $time = $timeOrYear;
        } else {
            $year = $timeOrYear;
            $time = "00:00";
        }

        return strtotime("$year-$monthNum-$day $time");
    }

    /** ============================
     *  Datei holen (für View / PDF)
     *  ============================ */
   public function getFileContents(string $path)
{
    $this->connect();

    $tmp = tmpfile();
    $meta = stream_get_meta_data($tmp);
    $local = $meta['uri'];

    // WARNUNGEN UNTERDRÜCKEN, nur Rückgabewert auswerten
    if (!@ftp_get($this->conn, $local, $path, FTP_BINARY)) {
        return false;
    }

    return file_get_contents($local);
}

    public function putFileContents(string $path, string $data): bool
    {
        $this->connect();

        $tmp = tmpfile();
        $m = stream_get_meta_data($tmp);
        file_put_contents($m['uri'], $data);

        return ftp_put($this->conn, $path, $m['uri'], FTP_BINARY);
    }

    public function delete(string $path): bool
    {
        $this->connect();

        // wenn Datei → delete()
        if (@ftp_delete($this->conn, $path)) {
            return true;
        }

        // wenn Ordner → erst Inhalt rekursiv löschen
        return $this->deleteDirRecursive($path);
    }

    /** ============================
     *  ORDNER REKURSIV LÖSCHEN
     *  ============================ */
    private function deleteDirRecursive(string $path): bool
    {
        $this->connect();

        $items = ftp_nlist($this->conn, $path);
        if (!is_array($items)) $items = [];

        foreach ($items as $item) {
            $name = basename($item);
            if ($name === '.' || $name === '..') continue;

            $sub = $path . '/' . $name;

            // Versuch Datei löschen
            if (@ftp_delete($this->conn, $sub)) {
                continue;
            }

            // sonst rekursiv
            $this->deleteDirRecursive($sub);
        }

        return @ftp_rmdir($this->conn, $path);
    }

    public function mkdir(string $path): bool
    {
        $this->connect();
        return ftp_mkdir($this->conn, $path) !== false;
    }

    public function rename(string $old, string $new): bool
    {
        $this->connect();
        return ftp_rename($this->conn, $old, $new);
    }

    public function chmod(string $path, int $mode): bool
    {
        $this->connect();
        if (function_exists('ftp_chmod')) {
            return ftp_chmod($this->conn, $mode, $path) !== false;
        }
        return false;
    }

    public function isDir(string $path): bool
    {
        $this->connect();

        $list = @ftp_rawlist($this->conn, $path);
        return ($list !== false && isset($list[0]) && str_starts_with($list[0], 'd'));
    }
}
