<?php
declare(strict_types=1);
require_once __DIR__ . '/../inc/config.php';
require_once __DIR__ . '/../inc/helpers.php';

$localFile = __DIR__ . '/../inc/config.local.php';
$dbFile    = __DIR__ . '/../data/mobiblog.sqlite';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $adminUser = trim($_POST['user'] ?? '');
  $adminPass = trim($_POST['pass'] ?? '');

  if ($adminUser === '' || $adminPass === '') {
    $err = "Bitte Benutzer und Passwort eingeben.";
  } else {
    // data/ anlegen
    $dataDir = dirname($dbFile);
    if (!is_dir($dataDir)) mkdir($dataDir, 0755, true);

    // SQLite DB erstellen
    $pdo = new PDO('sqlite:' . $dbFile);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    $pdo->exec("PRAGMA journal_mode = WAL;");
    $pdo->exec("PRAGMA foreign_keys = ON;");

    $pdo->exec("
      CREATE TABLE IF NOT EXISTS users (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        username TEXT NOT NULL UNIQUE,
        pass_hash TEXT NOT NULL,
        created_at INTEGER NOT NULL
      );

      CREATE TABLE IF NOT EXISTS posts (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        title TEXT NOT NULL,
        slug TEXT NOT NULL UNIQUE,
        content TEXT NOT NULL,
        status TEXT NOT NULL DEFAULT 'draft', -- draft|published
        published_at INTEGER,
        created_at INTEGER NOT NULL,
        updated_at INTEGER NOT NULL
      );

      CREATE INDEX IF NOT EXISTS idx_posts_status_pub ON posts(status, published_at);
    ");

    // Admin user
    $hash = password_hash($adminPass, PASSWORD_DEFAULT);
    $st = $pdo->prepare("INSERT INTO users(username, pass_hash, created_at) VALUES(?,?,?)");
    $st->execute([$adminUser, $hash, time()]);

    // config.local.php schreiben
    $cfg = "<?php\n".
           "declare(strict_types=1);\n".
           "define('SQLITE_PATH', " . var_export($dbFile, true) . ");\n";

    file_put_contents($localFile, $cfg);

    // uploads sicherstellen
    ensure_upload_dir();

    header('Location: ' . BASE_PATH . 'admin/login.php');
    exit;
  }
}
?>
<!doctype html>
<html lang="de">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Mobiblog Installer</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    body {
      background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
      min-height: 100vh;
      display: flex;
      align-items: center;
      padding: 20px;
    }
    .install-card {
      max-width: 600px;
      margin: 0 auto;
      width: 100%;
    }
  </style>
</head>
<body>
  <div class="container">
    <div class="install-card">
      <div class="card shadow-lg">
        <div class="card-body p-5">
          <div class="text-center mb-4">
            <h1 class="card-title h2">Mobiblog Installation</h1>
            <p class="text-muted">Richten Sie Ihr Blog in wenigen Schritten ein</p>
          </div>

          <?php if (!empty($err)): ?>
            <div class="alert alert-danger" role="alert">
              <i class="bi bi-exclamation-triangle-fill"></i> <?= e($err) ?>
            </div>
          <?php endif; ?>

          <?php if (is_file($localFile)): ?>
            <div class="alert alert-success" role="alert">
              <h5 class="alert-heading">
                <i class="bi bi-check-circle-fill"></i> Installation bereits abgeschlossen
              </h5>
              <p class="mb-0">Mobiblog ist bereits installiert und einsatzbereit.</p>
              <hr>
              <a href="<?= e(BASE_PATH) ?>admin/login.php" class="btn btn-success">
                Zum Admin-Login
              </a>
            </div>
            <div class="card bg-light mt-3">
              <div class="card-body">
                <h6 class="card-title">Sicherheitshinweis</h6>
                <p class="card-text small mb-0">
                  Sie können den <code>/install</code> Ordner nun aus Sicherheitsgründen löschen.
                </p>
              </div>
            </div>
          <?php else: ?>
            <form method="post">
              <div class="mb-3">
                <label for="user" class="form-label">Admin Benutzer</label>
                <input type="text" 
                       class="form-control" 
                       id="user" 
                       name="user" 
                       placeholder="Ihr Benutzername"
                       required 
                       autofocus>
              </div>

              <div class="mb-4">
                <label for="pass" class="form-label">Admin Passwort</label>
                <input type="password" 
                       class="form-control" 
                       id="pass" 
                       name="pass" 
                       placeholder="Ein sicheres Passwort"
                       required>
                <div class="form-text">Wählen Sie ein starkes Passwort für Ihr Admin-Konto.</div>
              </div>

              <div class="d-grid">
                <button type="submit" class="btn btn-primary btn-lg">
                  <i class="bi bi-download"></i> Jetzt installieren
                </button>
              </div>
            </form>

            <div class="card bg-light mt-4">
              <div class="card-body">
                <h6 class="card-title">Was passiert bei der Installation?</h6>
                <ul class="small mb-0">
                  <li>SQLite-Datenbank wird erstellt</li>
                  <li>Admin-Benutzer wird angelegt</li>
                  <li>Upload-Verzeichnis wird eingerichtet</li>
                  <li>Konfigurationsdatei wird erstellt</li>
                </ul>
              </div>
            </div>

            <div class="alert alert-info mt-3 mb-0" role="alert">
              <small>
                <i class="bi bi-info-circle"></i> 
                Nach der Installation können Sie den <code>/install</code> Ordner löschen.
              </small>
            </div>
          <?php endif; ?>
        </div>
      </div>
    </div>
  </div>

  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>