<?php

require_once __DIR__ . '/PdfHelper.php';

class FileSearch
{
    private string $root;
    private array $settings;

    public function __construct(string $root, array $settings)
    {
        $this->root     = rtrim($root, '/');
        $this->settings = $settings;
    }

    public function search(string $query): array
    {
        // ===============================
        // QUERY normalisieren
        // ===============================
        $query = urldecode($query);
        $query = trim($query);

        if ($query === '') {
            return [];
        }

        $results = [];
        $limit   = (int)($this->settings['search_limit'] ?? 50);

        // ===============================
        // ADMIN: erlaubte Endungen
        // ===============================
        $allowedExtensions = $this->settings['allowed_extensions'] ?? ['html','htm','php','txt','pdf'];

        if (is_string($allowedExtensions)) {
            $decoded = json_decode($allowedExtensions, true);
            if (is_array($decoded)) {
                $allowedExtensions = $decoded;
            } else {
                $allowedExtensions = explode(',', $allowedExtensions);
            }
        }

        $allowedExtensions = array_map(
            fn($e) => strtolower(ltrim(trim($e), '.')),
            $allowedExtensions
        );

        // ===============================
        // GET-FILTER: ?ext=pdf / html / php / txt
        // ===============================
        $filterExt = strtolower($_GET['ext'] ?? '');

        // ===============================
        // EXCLUDED FOLDERS aus Admin
        // ===============================
        $excludedFolders = $this->settings['excluded_folders'] ?? [];
        if (is_string($excludedFolders)) {
            $excludedFolders = json_decode($excludedFolders, true) ?: [];
        }

        $excludedFolders = array_map('trim', $excludedFolders);

        // ===============================
        // PROTECTED FILE PATTERNS
        // ===============================
        $protectedPatterns = $this->settings['protected_patterns'] ?? [];

        if (is_string($protectedPatterns) && str_starts_with(trim($protectedPatterns), '[')) {
            $tmp = json_decode($protectedPatterns, true);
            if (is_array($tmp)) {
                $protectedPatterns = $tmp;
            }
        }

        if (is_string($protectedPatterns)) {
            $protectedPatterns = explode(',', $protectedPatterns);
        }

        $protectedPatterns = array_map('trim', $protectedPatterns);
        $protectedPatterns = array_filter($protectedPatterns);
        $patternsNormalized = array_map('strtolower', $protectedPatterns);

        // ===============================
        // START FILE SCAN
        // ===============================
        $rii = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator(
                $this->root,
                FilesystemIterator::SKIP_DOTS | FilesystemIterator::FOLLOW_SYMLINKS
            )
        );

        foreach ($rii as $file) {

            if (count($results) >= $limit) {
                break;
            }

            if (!$file->isFile()) {
                continue;
            }

            $absFile = $file->getPathname();
            $name    = basename($absFile);
            $dir     = dirname($absFile);
            $ext     = strtolower(pathinfo($absFile, PATHINFO_EXTENSION));

            // ===============================
            // 1) ERSTE FILTERUNG: Dateityp
            // ===============================

            // Admin erlaubt?
            if (!in_array($ext, $allowedExtensions, true)) {
                continue;
            }

            // GET-Filter: wenn ext gesetzt ist, NUR diese Endung zulassen
            if ($filterExt !== '' && $ext !== $filterExt) {
                continue;
            }

            // ===============================
            // 2) FOLDER-EXCLUDE
            // ===============================
            foreach ($excludedFolders as $ex) {

                if ($ex === '') continue;

                $needle1 = '/' . ltrim($ex, '/');
                $needle2 = ltrim($ex, '/');

                if (
                    stripos($dir, $needle1) !== false ||
                    stripos($dir, $needle2) !== false
                ) {
                    continue 2;
                }
            }

            // ===============================
            // 3) PROTECTED-FILES
            // ===============================
            $lowerName = strtolower($name);
            $lowerPath = strtolower($absFile);

            foreach ($patternsNormalized as $pattern) {
                if ($pattern === '') continue;

                if (fnmatch($pattern, $lowerName, FNM_CASEFOLD)) continue 2;
                if (fnmatch($pattern, $lowerPath, FNM_CASEFOLD)) continue 2;
            }

            // ===============================
            // RELATIVE URL KORREKT BAUEN
            // ===============================
            $relFile = str_replace($this->root, '', $absFile);
            if ($relFile === '') {
                $relFile = '/';
            } elseif ($relFile[0] !== '/') {
                $relFile = '/' . $relFile;
            }

            // ===============================
            // 4) PDF-SCAN
            // ===============================
            if ($ext === 'pdf') {

                $text = PdfHelper::extractText($absFile);
                if (!$text) {
                    continue;
                }

                $pos = stripos($text, $query);
                if ($pos === false) {
                    continue;
                }

                $snippet = substr($text, max(0, $pos - 160), 320);
                $snippet = htmlspecialchars($snippet);
                $snippet = preg_replace(
                    "/(" . preg_quote($query, '/') . ")/i",
                    "<mark>$1</mark>",
                    $snippet
                );

                $results[] = [
                    'file'    => $absFile,
                    'url'     => $relFile,
                    'name'    => $name,
                    'type'    => 'pdf',
                    'preview' => $snippet
                ];

                continue;
            }

            // ===============================
            // 5) TEXT / HTML / PHP / TXT-SCAN
            // ===============================
            if (in_array($ext, ['html','htm','php','txt'], true)) {

                $content = @file_get_contents($absFile);
                if ($content === false) {
                    continue;
                }

                $pos = stripos($content, $query);
                if ($pos === false) {
                    continue;
                }

                $snippet = substr($content, max(0, $pos - 160), 320);

                $snippet = strip_tags($snippet);
                $snippet = htmlspecialchars($snippet);

                $snippet = preg_replace(
                    "/(" . preg_quote($query, '/') . ")/i",
                    "<mark>$1</mark>",
                    $snippet
                );

                $results[] = [
                    'file'    => $absFile,
                    'url'     => $relFile,
                    'name'    => $name,
                    'type'    => 'file',
                    'preview' => $snippet
                ];
            }
        }

        return $results;
    }
}
