<?php
ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);

require_once __DIR__ . '/FileSearch.php';
require_once __DIR__ . '/PdfHelper.php';

class SearchEngine
{
    private PDO $pdo;
    private array $settings = [];

    public function __construct(PDO $pdo)
    {
        $this->pdo = $pdo;
        $this->loadSettings();
    }

    public function getSettings(): array
    {
        return $this->settings;
    }

    private function loadSettings(): void
    {
        try {
            $stmt = $this->pdo->query("SELECT * FROM search_settings WHERE id = 1 LIMIT 1");
            $row  = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$row) {
                $this->settings = [
                    'result_mode'       => 'inline',
                    'search_limit'      => 50,
                    'widget_css'        => '',
                    'excluded_folders'  => '["/admin","/config"]',
                    'allowed_extensions'=> '["html","htm","php","txt","pdf"]',
                    'protected_patterns'=> 'config.php, *.db, .env, *.sql, *.key'
                ];
            } else {
                $this->settings = $row;
                if (empty($this->settings['result_mode'])) {
                    $this->settings['result_mode'] = 'inline';
                }
            }

        } catch (Throwable $e) {
            $this->settings = [
                'result_mode'       => 'inline',
                'search_limit'      => 50,
                'widget_css'        => '',
                'excluded_folders'  => '["/admin","/config"]',
                'allowed_extensions'=> '["html","htm","php","txt","pdf"]',
                'protected_patterns'=> 'config.php, *.db, .env, *.sql, *.key'
            ];
        }
    }

    private function searchDatabase(string $query): array
    {
        $out = [];

        try {
            $stmt = $this->pdo->prepare("
                SELECT path, title, content
                FROM pdf_index
                WHERE content LIKE :search
                LIMIT 50
            ");

            $stmt->execute([
                ':search' => '%' . $query . '%'
            ]);

            $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

        } catch (Throwable $e) {
            return [];
        }

        foreach ($rows as $row) {
            $text = $row['content'] ?? '';
            $pos  = stripos($text, $query);

            if ($pos === false) {
                $snippet = substr($text, 0, 300);
            } else {
                $snippet = substr($text, max(0, $pos - 80), 300);
            }

            $snippet = strip_tags($snippet);
            $snippet = htmlspecialchars($snippet);
            if ($query !== '') {
                $snippet = preg_replace(
                    "/(" . preg_quote($query, '/') . ")/i",
                    "<mark>$1</mark>",
                    $snippet
                );
            }

            $out[] = [
                'url'     => $row['path'],
                'name'    => $row['title'],
                'preview' => $snippet,
                'type'    => 'pdf'
            ];
        }

        return $out;
    }

    private function searchFilesystem(string $query): array
    {
        $basePath   = rtrim($_SERVER['DOCUMENT_ROOT'], '/');
        $fileSearch = new FileSearch($basePath, $this->settings);
        $fsResults  = $fileSearch->search($query);

        foreach ($fsResults as &$r) {
            if (!empty($r['file'])) {
                $r['url'] = str_replace($_SERVER['DOCUMENT_ROOT'], '', $r['file']);
            }

            if (empty($r['type'])) {
                $r['type'] = 'file';
            }
        }

        return $fsResults;
    }

    /**
     * HAUPTMETHODE – DB + FILESYSTEM + FILTER
     */
    public function search(string $query): array
    {
        $query = trim($query);
        if ($query === '') {
            return [];
        }

        // 1️⃣ Database-PDF Ergebnisse
        $dbResults = $this->searchDatabase($query);

        // 2️⃣ Filesystem Ergebnisse
        $fsResults = $this->searchFilesystem($query);

        // 3️⃣ zusammenführen
        $results = array_merge($dbResults, $fsResults);

        if (empty($results)) {
            return [];
        }

        /**
         * ⭐ NEU:
         * ext GET-Filter anwenden (pdf/html/php/txt)
         */
        $filterExt = strtolower($_GET['ext'] ?? '');

        if ($filterExt !== '') {
            $results = array_filter($results, function($r) use ($filterExt) {
                $ext = strtolower(pathinfo($r['file'] ?? $r['url'], PATHINFO_EXTENSION));
                return ($ext === $filterExt);
            });

            // Array neu indexieren
            $results = array_values($results);

            // WICHTIG: Wenn nach Filter keine Treffer → direkt zurückgeben
            if (empty($results)) {
                return [];
            }
        }

        /**
         *  Ranking (wie vorher)
         */
        $q = strtolower($query);

        usort($results, function($a, $b) use ($q) {
            $scoreA = 0;
            $scoreB = 0;

            $typeA = strtolower($a['type'] ?? '');
            $typeB = strtolower($b['type'] ?? '');

            $nameA = strtolower($a['name'] ?? '');
            $nameB = strtolower($b['name'] ?? '');

            $prevA = strip_tags(strtolower($a['preview'] ?? ''));
            $prevB = strip_tags(strtolower($b['preview'] ?? ''));

            if ($typeA === 'pdf') $scoreA += 10;
            if ($typeB === 'pdf') $scoreB += 10;

            $scoreA += substr_count($nameA, $q) * 3;
            $scoreB += substr_count($nameB, $q) * 3;

            $scoreA += substr_count($prevA, $q);
            $scoreB += substr_count($prevB, $q);

            return $scoreB <=> $scoreA;
        });

        return $results;
    }
}
