<?php
// ======================================================
// functions.php – Basisfunktionen für Wasserzeichen-Tool
// ======================================================

// Baut absolute Pfade aus relativen (nur intern genutzt)
function cfg_path(string $rel): string {
  return __DIR__ . '/' . ltrim($rel, '/');
}

// ------------------------------------------------------
// Konfigurationsdatei lesen/schreiben
// ------------------------------------------------------
function read_config(): array {
  $file = __DIR__ . '/config.json';
  if (!file_exists($file)) return [];
  $json = file_get_contents($file);
  $data = json_decode($json, true);
  return is_array($data) ? $data : [];
}

function save_config(array $cfg): void {
  $file = __DIR__ . '/config.json';
  file_put_contents($file, json_encode($cfg, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
}

// ------------------------------------------------------
// Pfad-Helfer (hoster-sicher, open_basedir-freundlich)
//
// Regeln:
// - Beginnt der Pfad mit "/"  -> relativ zum DOCUMENT_ROOT interpretieren
// - Beginnt der Pfad bereits mit DOCUMENT_ROOT -> unverändert
// - Windows-Laufwerk -> unverändert (falls lokal)
// - Sonst: relativ zum Skriptordner (__DIR__)
// ------------------------------------------------------
function normalize_path(string $dir): string {
  $dir = trim($dir);
  if ($dir === '') return '';

  $dir  = str_replace('\\', '/', $dir);
  $base = rtrim($_SERVER['DOCUMENT_ROOT'] ?? __DIR__, '/');

  // Bereits absolut unterhalb von DOCUMENT_ROOT?
  if (str_starts_with($dir, $base . '/')) {
    return rtrim($dir, '/');
  }

  // Führender Slash -> als DOCROOT-relativ interpretieren
  if (str_starts_with($dir, '/')) {
    return rtrim($base . $dir, '/');
  }

  // Windows-Laufwerk (lokale Umgebung)
  if (preg_match('#^[A-Za-z]:/#', $dir)) {
    return rtrim($dir, '/');
  }

  // Relativer Pfad -> relativ zum Scriptordner
  return rtrim(__DIR__ . '/' . ltrim($dir, '/'), '/');
}

// ------------------------------------------------------
// Alle unterstützten Bilddateien eines Ordners
// (glob mit GLOB_BRACE; falls Hoster das blockt, auf scandir umbauen)
// ------------------------------------------------------
function list_images(string $dir): array {
  $path = normalize_path($dir);
  if (!is_dir($path)) return [];

  $imgs = glob($path . '/*.{jpg,jpeg,png,gif,webp,JPG,JPEG,PNG,GIF,WEBP}', GLOB_BRACE);
  if ($imgs === false) $imgs = []; // Sicherheitsfallback

  // Nur lesbare Dateien und echte Dateien
  $imgs = array_values(array_filter($imgs, function ($f) {
    return is_file($f) && is_readable($f);
  }));

  sort($imgs, SORT_NATURAL | SORT_FLAG_CASE);
  return $imgs;
}

// ------------------------------------------------------
// Sicherer Dateiname (kein Pfad, nur Basename)
// ------------------------------------------------------
function safe_name(string $p): string {
  return basename(str_replace('\\', '/', $p));
}
