<?php
session_start();

// Konfiguration
// WICHTIG: Ersetzen Sie den Platzhalter-Hash mit Ihrem eigenen, generierten Hash
$admin_password_hash = '$2y$10$YXHcam23QmOp4AM1eD8RxObh2T.FDto04lrt2V2o1OOS78L3AmJkK';
$entries_file = 'entries.json';
$pending_file = 'pending.json';

// Paginierung-Einstellungen
$entries_per_page = 5; // Anzahl der Einträge pro Seite

// Hilfsfunktionen
function loadEntries($file) {
    if (!file_exists($file)) {
        return [];
    }
    $content = file_get_contents($file);
    return json_decode($content, true) ?: [];
}

function saveEntries($entries, $file) {
    file_put_contents($file, json_encode($entries, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

// Verarbeitung der Formulareingaben
$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'admin_login':
                if (password_verify($_POST['password'], $admin_password_hash)) {
                    $_SESSION['admin'] = true;
                    $message = 'Erfolgreich als Administrator angemeldet.';
                } else {
                    $error = 'Falsches Passwort. Zugang verweigert.';
                }
                break;
                
            case 'admin_logout':
                unset($_SESSION['admin']);
                $message = 'Erfolgreich abgemeldet.';
                break;
                
            case 'approve_entry':
                if (isset($_SESSION['admin'])) {
                    $pending = loadEntries($pending_file);
                    $approved = loadEntries($entries_file);
                    $entry_id = $_POST['entry_id'];
                    
                    foreach ($pending as $key => $entry) {
                        if ($entry['id'] === $entry_id) {
                            $approved[] = $entry;
                            unset($pending[$key]);
                            break;
                        }
                    }
                    
                    saveEntries(array_values($pending), $pending_file);
                    saveEntries($approved, $entries_file);
                    $message = 'Eintrag wurde erfolgreich freigegeben.';
                } else {
                    $error = 'Nicht autorisiert.';
                }
                break;
                
            case 'reject_entry':
                if (isset($_SESSION['admin'])) {
                    $pending = loadEntries($pending_file);
                    $entry_id = $_POST['entry_id'];
                    
                    foreach ($pending as $key => $entry) {
                        if ($entry['id'] === $entry_id) {
                            unset($pending[$key]);
                            break;
                        }
                    }
                    
                    saveEntries(array_values($pending), $pending_file);
                    $message = 'Eintrag wurde abgelehnt und entfernt.';
                } else {
                    $error = 'Nicht autorisiert.';
                }
                break;
                
            case 'delete_entry':
                if (isset($_SESSION['admin'])) {
                    $entries = loadEntries($entries_file);
                    $entry_id = $_POST['entry_id'];
                    
                    foreach ($entries as $key => $entry) {
                        if ($entry['id'] === $entry_id) {
                            unset($entries[$key]);
                            break;
                        }
                    }
                    
                    saveEntries(array_values($entries), $entries_file);
                    $message = 'Eintrag wurde permanent gelöscht.';
                } else {
                    $error = 'Nicht autorisiert.';
                }
                break;
                
            case 'approve_all':
                if (isset($_SESSION['admin'])) {
                    $pending = loadEntries($pending_file);
                    $approved = loadEntries($entries_file);
                    
                    $count = count($pending);
                    
                    if ($email_enabled && class_exists('EmailConfig') && EmailConfig::SEND_AUTHOR_CONFIRMATIONS) {
                        try {
                            $mailer = new GuestbookMailer(EmailConfig::ADMIN_EMAIL, EmailConfig::SITE_NAME);
                            foreach ($pending as $entry) {
                                if (!empty($entry['email'])) {
                                    $mailer->notifyEntryApproved($entry);
                                }
                            }
                        } catch (Exception $e) {
                            error_log("Gästebuch E-Mail Fehler: " . $e->getMessage());
                        }
                    }
                    
                    $approved = array_merge($approved, $pending);
                    
                    saveEntries($approved, $entries_file);
                    saveEntries([], $pending_file);
                    $message = "$count Einträge wurden alle auf einmal freigegeben.";
                } else {
                    $error = 'Nicht autorisiert.';
                }
                break;
                
            case 'reject_all':
                if (isset($_SESSION['admin'])) {
                    $count = count(loadEntries($pending_file));
                    saveEntries([], $pending_file);
                    $message = "$count wartende Einträge wurden alle abgelehnt.";
                } else {
                    $error = 'Nicht autorisiert.';
                }
                break;
        }
    }
}

// Daten laden
$entries = loadEntries($entries_file);
$pending = loadEntries($pending_file);
$isAdmin = isset($_SESSION['admin']);

// Paginierung berechnen
$total_entries = count($entries);
$total_pages = ceil($total_entries / $entries_per_page);
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
if ($page < 1) {
    $page = 1;
}
if ($page > $total_pages && $total_pages > 0) {
    $page = $total_pages;
}
$start_index = ($page - 1) * $entries_per_page;

// Nur die Einträge für die aktuelle Seite extrahieren
$paged_entries = array_slice(array_reverse($entries), $start_index, $entries_per_page);

?>

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gästebuch - Administration</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap-icons/1.10.0/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        .admin-header {
            background: linear-gradient(135deg, #dc3545 0%, #6f42c1 100%);
        }
        .pending-card {
            border-left: 4px solid #ffc107;
            background-color: #fff9e6;
        }
        .approved-card {
            border-left: 4px solid #28a745;
        }
        .stats-card {
            background: linear-gradient(135deg, #17a2b8 0%, #007bff 100%);
            color: white;
        }
        .login-card {
            box-shadow: 0 15px 35px rgba(0,0,0,0.1);
            border: none;
        }
    </style>
    <script>
        window.addEventListener("load", fenster);
        window.addEventListener("resize", fenster);
        function fenster() {
          if (parent.document.getElementsByTagName("iframe")[0]) {
            parent.document.getElementsByTagName("iframe")[0].style.height =
              document.getElementsByTagName("html")[0].offsetHeight +'px';
          }
        }
    </script>
</head>
<body class="bg-light">
    <div class="admin-header text-white py-4 mb-4">
        <div class="container">
            <div class="row align-items-center">
                <div class="col">
                    <h1 class="h3 mb-0">
                        <i class="bi bi-shield-check"></i> Gästebuch Administration
                    </h1>
                </div>
                <div class="col-auto">
                    <a href="index.php" class="btn btn-light">
                        <i class="bi bi-arrow-left"></i> Zurück zum Gästebuch
                    </a>
                </div>
            </div>
        </div>
    </div>

    <div class="container">
        <?php if ($message): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <i class="bi bi-check-circle"></i> <?= $message ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if ($error): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <i class="bi bi-exclamation-circle"></i> <?= $error ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if (!$isAdmin): ?>
            <div class="row justify-content-center">
                <div class="col-md-6">
                    <div class="card login-card">
                        <div class="card-header bg-danger text-white text-center">
                            <h4 class="mb-0"><i class="bi bi-lock"></i> Administrator Login</h4>
                        </div>
                        <div class="card-body p-5">
                            <form method="POST">
                                <input type="hidden" name="action" value="admin_login">
                                <div class="mb-4">
                                    <label for="password" class="form-label fw-semibold">Passwort</label>
                                    <input type="password" name="password" id="password" 
                                           class="form-control form-control-lg" 
                                           placeholder="Admin-Passwort eingeben" required>
                                </div>
                                <div class="d-grid">
                                    <button type="submit" class="btn btn-danger btn-lg">
                                        <i class="bi bi-key"></i> Anmelden
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        <?php else: ?>
            <div class="row mb-4">
                <div class="col-md-4 mb-3">
                    <div class="card stats-card">
                        <div class="card-body text-center">
                            <i class="bi bi-hourglass-split display-4 mb-2"></i>
                            <h3 class="mb-0"><?= count($pending) ?></h3>
                            <p class="mb-0">Wartende Einträge</p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4 mb-3">
                    <div class="card bg-success text-white">
                        <div class="card-body text-center">
                            <i class="bi bi-check-circle display-4 mb-2"></i>
                            <h3 class="mb-0"><?= count($entries) ?></h3>
                            <p class="mb-0">Freigegebene Einträge</p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4 mb-3">
                    <div class="card bg-info text-white">
                        <div class="card-body text-center">
                            <i class="bi bi-graph-up display-4 mb-2"></i>
                            <h3 class="mb-0"><?= count($pending) + count($entries) ?></h3>
                            <p class="mb-0">Einträge gesamt</p>
                        </div>
                    </div>
                </div>
            </div>

            <div class="row mb-4">
                <div class="col">
                    <div class="card">
                        <div class="card-header bg-secondary text-white">
                            <h5 class="mb-0"><i class="bi bi-gear"></i> Admin-Funktionen</h5>
                        </div>
                        <div class="card-body">
                            <div class="btn-group me-2" role="group">
                                <?php if (!empty($pending)): ?>
                                    <form method="POST" class="d-inline">
                                        <input type="hidden" name="action" value="approve_all">
                                        <button type="submit" class="btn btn-success" 
                                                onclick="return confirm('Alle wartenden Einträge freigeben?')">
                                            <i class="bi bi-check-all"></i> Alle freigeben
                                        </button>
                                    </form>
                                    <form method="POST" class="d-inline">
                                        <input type="hidden" name="action" value="reject_all">
                                        <button type="submit" class="btn btn-warning" 
                                                onclick="return confirm('Alle wartenden Einträge ablehnen?')">
                                            <i class="bi bi-x-circle"></i> Alle ablehnen
                                        </button>
                                    </form>
                                <?php endif; ?>
                                <form method="POST" class="d-inline">
                                    <input type="hidden" name="action" value="admin_logout">
                                    <button type="submit" class="btn btn-outline-danger">
                                        <i class="bi bi-box-arrow-right"></i> Abmelden
                                    </button>
                                </form>
                                <div class="col-auto">
                    <a href="backup.php" class="btn btn-info">
                        <i class="bi bi-floppy"></i> Backup erstellen
                    </a>
                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <?php if (!empty($pending)): ?>
                <div class="card mb-4">
                    <div class="card-header bg-warning text-dark">
                        <h5 class="mb-0">
                            <i class="bi bi-hourglass-split"></i> 
                            Wartende Einträge zur Freischaltung (<?= count($pending) ?>)
                        </h5>
                    </div>
                    <div class="card-body">
                        <?php foreach ($pending as $entry): ?>
                            <div class="card mb-3 pending-card">
                                <div class="card-body">
                                    <div class="row">
                                        <div class="col-md-8">
                                            <h6 class="card-title text-primary">
                                                <i class="bi bi-person"></i> <?= htmlspecialchars($entry['name']) ?>
                                            </h6>
                                            <?php if (!empty($entry['email'])): ?>
                                                <p class="card-text mb-1">
                                                    <small class="text-muted">
                                                        <i class="bi bi-envelope"></i> <?= htmlspecialchars($entry['email']) ?>
                                                    </small>
                                                </p>
                                            <?php endif; ?>
                                            <p class="card-text"><?= nl2br(htmlspecialchars($entry['comment'])) ?></p>
                                        </div>
                                        <div class="col-md-4 text-end">
                                            <small class="text-muted d-block mb-3">
                                                <i class="bi bi-calendar"></i> <?= $entry['date'] ?>
                                            </small>
                                            <div class="btn-group-vertical d-grid gap-2" role="group">
                                                <form method="POST" class="d-inline">
                                                    <input type="hidden" name="action" value="approve_entry">
                                                    <input type="hidden" name="entry_id" value="<?= $entry['id'] ?>">
                                                    <button type="submit" class="btn btn-success">
                                                        <i class="bi bi-check-lg"></i> Freigeben
                                                    </button>
                                                </form>
                                                <form method="POST" class="d-inline">
                                                    <input type="hidden" name="action" value="reject_entry">
                                                    <input type="hidden" name="entry_id" value="<?= $entry['id'] ?>">
                                                    <button type="submit" class="btn btn-danger"
                                                            onclick="return confirm('Eintrag wirklich ablehnen?')">
                                                        <i class="bi bi-x-lg"></i> Ablehnen
                                                    </button>
                                                </form>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            <?php else: ?>
                <div class="alert alert-info">
                    <i class="bi bi-info-circle"></i> Keine wartenden Einträge vorhanden.
                </div>
            <?php endif; ?>

            <div class="card">
                <div class="card-header bg-success text-white">
                    <h5 class="mb-0">
                        <i class="bi bi-check-circle"></i> 
                        Freigegebene Einträge (<?= $total_entries ?>)
                    </h5>
                </div>
                <div class="card-body">
                    <?php if (empty($entries)): ?>
                        <p class="text-muted text-center py-3">
                            <i class="bi bi-journal-x display-4"></i><br>
                            Noch keine freigegebenen Einträge vorhanden.
                        </p>
                    <?php else: ?>
                        <?php foreach ($paged_entries as $entry): ?>
                            <div class="card mb-3 approved-card">
                                <div class="card-body">
                                    <div class="row">
                                        <div class="col-md-10">
                                            <h6 class="card-title text-success">
                                                <i class="bi bi-person-check"></i> <?= htmlspecialchars($entry['name']) ?>
                                            </h6>
                                            <?php if (!empty($entry['email'])): ?>
                                                <p class="card-text mb-1">
                                                    <small class="text-muted">
                                                        <i class="bi bi-envelope"></i> <?= htmlspecialchars($entry['email']) ?>
                                                    </small>
                                                </p>
                                            <?php endif; ?>
                                            <p class="card-text"><?= nl2br(htmlspecialchars($entry['comment'])) ?></p>
                                            <small class="text-muted">
                                                <i class="bi bi-calendar"></i> <?= $entry['date'] ?>
                                            </small>
                                        </div>
                                        <div class="col-md-2 text-end">
                                            <form method="POST" class="d-inline">
                                                <input type="hidden" name="action" value="delete_entry">
                                                <input type="hidden" name="entry_id" value="<?= $entry['id'] ?>">
                                                <button type="submit" class="btn btn-outline-danger btn-sm" 
                                                        onclick="return confirm('Eintrag permanent löschen?')">
                                                    <i class="bi bi-trash"></i> Löschen
                                                </button>
                                            </form>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>

                        <?php if ($total_pages > 1): ?>
                            <nav aria-label="Eintrag-Paginierung">
                                <ul class="pagination justify-content-center mt-4">
                                    <?php if ($page > 1): ?>
                                        <li class="page-item">
                                            <a class="page-link" href="?page=<?= $page - 1 ?>" aria-label="Vorherige">
                                                <span aria-hidden="true">&laquo;</span>
                                            </a>
                                        </li>
                                    <?php endif; ?>
                                    <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                                        <li class="page-item <?= ($i == $page) ? 'active' : '' ?>">
                                            <a class="page-link" href="?page=<?= $i ?>"><?= $i ?></a>
                                        </li>
                                    <?php endfor; ?>
                                    <?php if ($page < $total_pages): ?>
                                        <li class="page-item">
                                            <a class="page-link" href="?page=<?= $page + 1 ?>" aria-label="Nächste">
                                                <span aria-hidden="true">&raquo;</span>
                                            </a>
                                        </li>
                                    <?php endif; ?>
                                </ul>
                            </nav>
                        <?php endif; ?>

                    <?php endif; ?>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <script src="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/js/bootstrap.bundle.min.js"></script>
    <script>
    (function () {
      function scrollParentToGastbuch() {
        try {
          if (window.parent && window.parent !== window) {
            var parentDoc = window.parent.document;
            setTimeout(function () {
              var anchor = parentDoc.getElementById('Gastbuch');
              if (anchor) {
                anchor.scrollIntoView({ behavior: 'smooth', block: 'start' });
              }
            }, 50);
          }
        } catch (e) {
          console.warn('Scroll zu #Gastbuch nicht möglich:', e);
        }
      }

      window.addEventListener('load', scrollParentToGastbuch);

      document.addEventListener('click', function (e) {
        var link = e.target.closest('a.page-link');
        if (!link) return;

        var href = link.getAttribute('href') || '';
        if (href.indexOf('?page=') !== -1) {
          scrollParentToGastbuch();
        }
      });
    })();
    </script>
</body>
</html>